/****************************************************************************

(c) 2017 by STEMMER IMAGING GmbH
Project: GenApi-C
Author:  Sascha Dorenbeck

License: This file is published under the license of the EMVA GenICam  Standard Group.
A text file describing the legal terms is included in  your installation as 'GenICam_license.pdf'.
If for some reason you are missing  this file please contact the EMVA or visit the website
(http://www.genicam.org) for a full copy.

THIS SOFTWARE IS PROVIDED BY THE EMVA GENICAM STANDARD GROUP "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE EMVA GENICAM STANDARD  GROUP
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,  SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT  LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,  DATA, OR PROFITS;
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY  THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE  OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

****************************************************************************/


#ifndef GENAPI_C_IFACE_H
#define GENAPI_C_IFACE_H


#include <stdint.h>
#include <stddef.h>
#include "GenApiCVersion.h"


#if defined (_WIN32) && (defined (_M_IX86) || defined (__i386__)) 
#   define GENAPI_CALLTYPE __cdecl 
#else 
#   define GENAPI_CALLTYPE /* default */ 
#endif


/* Error Status Type */
typedef int32_t GenApiError;


/* Error Status Codes */
typedef enum
{
    GenApiSuccess = 0,
    GenApiErrorGeneric = -1,
    GenApiErrorBadAlloc = -2,
    GenApiErrorInvalidArgument = -3,
    GenApiErrorOutOfRange = -4,
    GenApiErrorProperty = -5,
    GenApiErrorRuntime = -6,
    GenApiErrorLogic = -7,
    GenApiErrorAccess = -8,
    GenApiErrorTimeout = -9
} GenApiErrorCodes;


/*! Exported functions */
#define GENAPI_CALL extern "C" GenApiError GENAPI_CALLTYPE
/*! Function pointers for dynamic loading */
#define GENAPI_PCALL(function) typedef GenApiError( GENAPI_CALLTYPE *function )


/* Node Property Data Types */
typedef enum
{
    GenApiUnknownType = -1,
    GenApiString,
    GenApiInt64,
    GenApiFloat64,
    GenApiBuffer
} GenApiDataType;


/*
 * Common Public Functions
 */

/*!
 * \brief Gets the full version of the library.
 *
 * All parameters are optional and can be set to \b NULL.
 *
 * \param [out] pMajor Pointer to variable to receive the major version number.
 * \param [out] pMinor Pointer to variable to receive the minor version number.
 * \param [out] pSubMinor Pointer to variable to receive the sub-minor version
 *                  number.
 * \return Always #GenApiSuccess.
 */
GENAPI_PCALL(PGenApiGetVersion)(uint16_t *pMajor, uint16_t *pMinor, uint16_t *pSubMinor);


/*!
 * \brief Gets the last error \a Reason.
 *
 * This function is thread-local and intended to be called right after an error
 * occurred calling one of this API's functions (return value was not
 * #GenApiSuccess). This value will only be updated when a new error occurs.
 *
 * A size query will also add the C string nul-termination. The nul-termination
 * will also be written if enough space is available.
 *
 * \note
 * An error while using this function won't affect the last \a Reason text.
 *
 * \param [in] Reason Pointer to buffer that is filled with the string data;
 *                  \b NULL for size query.
 * \param [in,out] pReasonSize The size in characters of \a Reason as input;
 *                  will receive the required (if \a Reason is \b NULL) size or
 *                  actually written size.
 * \return
 * - #GenApiSuccess if accessed successfully
 * - #GenApiErrorInvalidArgument if \a pReasonSize is \b NULL.
 */
GENAPI_PCALL(PGenApiGetLastErrorString)(char *Reason, size_t *pReasonSize);


/*
 * Public Node Types and Functions 
 */

/* Node Types */
typedef enum
{
    GenApiUndefinedNode     = 0x0000,

    GenApiNode              = 0x0001,
    GenApiValueNode         = 0x0002 | GenApiNode,

    GenApiRegisterNode      = 0x0004 | GenApiValueNode,
    GenApiIntegerNode       = 0x0010 | GenApiValueNode,
    GenApiIntRegNode        = GenApiIntegerNode | GenApiRegisterNode,
    GenApiFloatNode         = 0x0020 | GenApiValueNode,
    GenApiFloatRegNode      = GenApiFloatNode | GenApiRegisterNode,
    GenApiStringNode        = 0x0040 | GenApiValueNode,
    GenApiStringRegNode     = GenApiStringNode | GenApiRegisterNode,
    GenApiBooleanNode       = 0x0080 | GenApiValueNode,
    GenApiCommandNode       = 0x00A0 | GenApiValueNode,
    GenApiEnumerationNode   = 0x00C0 | GenApiValueNode,
    GenApiEnumEntryNode     = 0x00E0 | GenApiValueNode,
    GenApiCategoryNode      = 0x0100 | GenApiValueNode,
    GenApiPortNode          = 0x0200 | GenApiNode
} GenApiNodeType;


/* Node Properties */
typedef enum
{
    GenApiNodeName,
    GenApiNodeAccessMode,
    GenApiNodeImposedAccessMode,
    GenApiNodeVisibility,
    GenApiNodeMaxValue,
    GenApiNodeMinValue,
    GenApiNodeIncrement,
    GenApiNodeUnit,
    GenApiNodeDescription,
    GenApiNodeToolTip,
    GenApiNodeRepresentation,
    GenApiNodeIsStreamable,
    GenApiNodeIsDeprecated,
    GenApiNodePollingTime,
    GenApiNodeDisplayName,
    GenApiNodeRegisterAddress,
    GenApiNodeRegisterLength,
    GenApiNodeEventID,
    GenApiNodeIsFeature,
    GenApiNodeCachingMode,
    GenApiNodePrincipalInterfaceType,
    GenApiNodeDisplayPrecision,
    GenApiNodeDocuURL,
    GenApiNodeDisplayNotation,
    GenApiNodeIncMode,
    GenApiNodeIsSelfClearing,
    GenApiNodeNamespace,
    GenApiNodeIsSelector,
    GenApiNodeAlias,
    GenApiNodeCastAlias
} GenApiNodeProperty;


/* Access Modes */
typedef enum  /* TODO: make source of this enum the same as C++ GenApi */
{
    GenApiNotImplemented = 0x10,
    GenApiNotAvailable = 0x00,
    GenApiReadOnly = 0x01,
    GenApiWriteOnly = 0x02,
    GenApiReadWrite = GenApiReadOnly | GenApiWriteOnly
} GenApiAccessMode;


/* Node Visibility */
typedef enum  /* TODO: make source of this enum the same as C++ GenApi */
{
    GenApiInvisible,
    GenApiBeginner,
    GenApiExpert,
    GenApiGuru
} GenApiVisibility;


/* Node Value read/write flags */
typedef enum
{
    GenApiValueDoNotVerify = 0x00,
    GenApiValueDoVerify = 0x01,
    GenApiIgnoreCache = 0x02,
    GenApiRawData = 0x04,  // Get or set a string that represents a raw numeric value.

    GenApiReadValueDefault = GenApiValueDoNotVerify,
    GenApiWriteValueDefault = GenApiValueDoVerify
} GenApiNodeValueFlags;


/* Node Link types */
typedef enum  // They are used as a parameter of the NodeGetLinks function.
{
    GenApiFeatureNodes,
    GenApiEntries,
    GenApiSymbolics,
    GenApiSelectingNodes,
    GenApiSelectedNodes,
    GenApiReadingNodes,
    GenApiWritingNodes,
    GenApiInvalidatingNodes,
    GenApiDependingNodes,
    GenApiTerminalNodes,
    GenApiValueChangingNodes,
    GenApiParentNodes
} GenApiNodeLinkType;


/* Representations */
typedef enum  // They are equivalent to the C++'s ERepresentation.
{
    GenApiLinear = 0,
    GenApiLogarithmic,
    GenApiBoolean,
    GenApiPureNumber,
    GenApiHexNumber,
    GenApiIPV4Address,
    GenApiMACAddress,
    GenApiUndefinedRepresentation
} GenApiRepresentation;


/* Caching Mode */
typedef enum  // They are equivalent to the C++'s ECachingMode.
{
    GenApiNoCache = 0,
    GenApiWriteThrough,
    GenApiWriteAround,
    GenApiUndefinedCachingMode
} GenApiCachingMode;


/* Principal Interface Type */
typedef enum  // They are equivalent to the C++'s EInterfaceType.
{
    GenApiIValue = 0,
    GenApiIBase,
    GenApiIInteger,
    GenApiIBoolean,
    GenApiICommand,
    GenApiIFloat,
    GenApiIString,
    GenApiIRegister,
    GenApiICategory,
    GenApiIEnumeration,
    GenApiIEnumEntry,
    GenApiIPort
} GenApiPrincipalInterfaceType;


/* Display Notation */
typedef enum  // They are equivalent to the C++'s EDisplayNotation.
{
    GenApiAutomatic = 0,
    GenApiFixed,
    GenApiScientific,
    GenApiUndefinedEDisplayNotation
} GenApiDisplayNotation;


/* Inc Mode */
typedef enum  // They are equivalent to the C++'s EIncMode.
{
    GenApiNoIncrement = 0,
    GenApiFixedIncrement,
    GenApiListIncrement
} GenApiIncMode;


/* Namespace */
typedef enum  // They are equivalent to the C++'s ENameSpace.
{
    GenApiCustom = 0,
    GenApiStandard,
    GenApiUndefinedNameSpace
} GenApiNamespace;


/* Handle for Nodes */
typedef struct
{
    void *Handle;
    GenApiNodeType NodeType;
} GenApiNodeHandle;

/* Callback events */
typedef enum
{
    GenApiNodeOnUpdated = 0
} GenApiNodeCallbackEvent;

/* Public Node Functions */

/*!
 * \brief Gets the value of the given \a hNode.
 * 
 * If you want to query the type of the value, simply set \a pType to
 * #GenApiUnknownType.
 *
 * For the #GenApiString type a size query will also add the C string
 * nul-termination. The nul-termination will also be written if enough space is
 * available.
 * 
 * \param [in] hNode Node handle of object to get the value for.
 * \param [in] Flags Configure the read, e.g. #GenApiReadValueDefault.
 * \param [in,out] pType Hint to get the value as a certain type; if input was
 *              #GenApiUnknownType this will receive the actual value type.
 * \param [in] pValue Pointer to buffer to be filled with the value; \b NULL
 *              for size query.
 * \param [in,out] pValueSize The size in bytes of \a pValue as input; will
 *                  receive the required (if \a pValue is \b NULL) size or
 *                  actually written size.
 * \return
 * - #GenApiSuccess if read successfully
 * - #GenApiErrorInvalidArgument if \a hNode, \a pType or \a pValueSize is 
 *   \b NULL.
 * - #GenApiErrorGeneric on other errors.
 * \todo investigate possible exceptions
 */
GENAPI_PCALL(PGenApiNodeGetValue)(GenApiNodeHandle hNode, int Flags, GenApiDataType *pType, void *pValue, size_t *pValueSize);


/*!
 * \brief Sets the value of the given node.
 * 
 * For \a Type #GenApiUnknownType is never supported. For #GenApiString a
 * nul-terminated C string is not a must. If you don't want to have the nul
 * character being set, don't include it in the \a ValueSize.
 * 
 * \param [in] hNode Node handle of object to set the value for.
 * \param [in] Flags Configure the write, e.g. #GenApiWriteValueDefault.
 * \param [in] pValue Pointer to buffer to be written to the node; may be 
 *              \b NULL for commands.
 * \param [in] ValueSize Size in bytes of \a pValue.
 * \return
 * - #GenApiSuccess if written successfully
 * - #GenApiErrorInvalidArgument if \a hNode, \a pValue is \b NULL or 
 *   \a ValueSize is 0.
 * - #GenApiErrorGeneric on other errors.
 * \todo investigate possible exceptions
 */
GENAPI_PCALL(PGenApiNodeSetValue)(GenApiNodeHandle hNode, int Flags, GenApiDataType Type, const void *pValue, size_t ValueSize);


/*!
 * \brief Gets a \a Property from the node identified by \a hNode.
 * 
 * If you want to query the type of the property, simply set \a pType to 
 * #GenApiUnknownType.
 * 
 * For the #GenApiString type a size query will also add the C string 
 * nul-termination. The nul-termination will also be written if enough space is
 * available.
 * 
 * \param [in] hNode Handle of node object to access.
 * \param [in] Property Identifies the property to access.
 * \param [in,out] pType Non \b NULL hint of which data type to receive; will
 *                  receive written data type.
 * \param [in] pValue Pointer to buffer that is filled with requested data;
 *                  \b NULL for size query.
 * \param [in,out] pValueSize The size in bytes of \a pValue as input; will 
 *                  receive the required (if \a pValue is \b NULL) size or
 *                  actually written size.
 * \return
 * - #GenApiSuccess if property was read successfully.
 * - #GenApiErrorInvalidArgument if \a hNode is \b NULL, \a Property is 
 *      unknown, \a pType is \b NULL or incompatible.
 */
GENAPI_PCALL(PGenApiNodeGetProperty)(GenApiNodeHandle hNode, GenApiNodeProperty Property, GenApiDataType *pType, void *pValue, size_t *pValueSize);


/*!
 * \brief Gets a list of node names with the relation defined by \a Type.
 * 
 * The \a pNameList is filled with the names of all nodes in \a Type relation
 * of \a hNode. Each name 0-terminated. The whole list is also terminated with
 * a 0-byte resulting in double-0s at the end.
 * 
 * \param [in] hNode Node handle of object to query node names for.
 * \param [in] Type Type of links to query, e.g. #GenApiFeatureNodes.
 * \param [in] pNameList Pointer to buffer to be filled with name list; \b NULL
 *              for size query.
 * \param [in,out] pNameListSize The size in bytes of \a pNameList as input; 
 *                  will receive the required (if \a pNameList is \b NULL) size 
 *                  or actually written size.
 * \return
 * - #GenApiSuccess if written successfully
 * - #GenApiErrorInvalidArgument if \a hNodeMap or \a pNameListSize is
 *   \b NULL.
 * - #GenApiErrorGeneric on other errors.
 * \todo investigate possible exceptions
 */
GENAPI_PCALL(PGenApiNodeGetLinks)(GenApiNodeHandle hNode, GenApiNodeLinkType Type, char *pNameList, size_t *pNameListSize);


/*
 * Public Transport Layer Provider Port types and functions.
 */

/* Handle for the Transport Layer Provider port */
typedef struct
{
    void *Handle;
} GenApiTLPortHandle;


/*!
 * \brief Callback that is called when a register performs a read operation.
 * 
 * The \a hPort was given to #PGenApiNodeMapConnectPort.
 * 
 * \param [in] hPort User specified pointer; e.g. an object's this pointer.
 * \param [in] Address Device register address to read from.
 * \param [in] pBuffer Pointer to buffer containing the data to be written to
 *              the device.
 * \param [in] BufferLength Length of \a pBuffer in bytes (number of bytes to
 *              write).
 * \return One of #GenApiErrorCodes entries, normally
 * - #GenApiSuccess if read was successful.
 * - #GenApiErrorAccess if access was forbidden or not possible
 * - #GenApiErrorTimeout if the device didn't answer in time.
 */
GENAPI_PCALL(PGenApiTLPortRead)(GenApiTLPortHandle hPort, int64_t Address, void *pBuffer, int64_t BufferLength);


/*!
 * \brief Callback that is called when a register performs a write operation.
 * 
 * The \a hPort pointer was given to #PGenApiNodeMapConnectPort.
 * 
 * \param [in] hPort User specified pointer; e.g. an object's this pointer.
 * \param [in] Address Device register address to write to.
 * \param [in] pBuffer Pointer to buffer to contain the data read from the 
 *              device.
 * \param [in] BufferLength Length of \a pBuffer in bytes (number of bytes to 
 *              read).
 * \return One of #GenApiErrorCodes entries, normally
 * - #GenApiSuccess if write was successful.
 * - #GenApiErrorAccess if access was forbidden or not possible
 * - #GenApiErrorTimeout if the device didn't answer in time.
 */
GENAPI_PCALL(PGenApiTLPortWrite)(GenApiTLPortHandle hPort, int64_t Address, const void *pBuffer, int64_t BufferLength);


/*!
 * \brief Callback that is called when a port is asked for its access mode.
 * 
 * The \a hPort pointer was given to #PGenApiNodeMapConnectPort.
 * 
 * \param [in] hPort User specified pointer; e.g. an object's this pointer.
 * \param [out] pMode Variable to receive the access mode of the \a hPort.
 * \return One of #GenApiErrorCodes, normally
 * - #GenApiSuccess if query was successful.
 * - #GenApiErrorInvalidArgument if \a hPort was invalid or \a pMode is \b NULL.
 */
GENAPI_PCALL(PGenApiTLPortGetAccessMode)(GenApiTLPortHandle hPort, GenApiAccessMode *pMode);


/*
 * Public NodeMap Types and Functions
 */

 /* Handle for the NodeMap */
typedef struct
{
    void *Handle;
} GenApiNodeMapHandle;


/* Flags to configure NodeMap creation */
typedef enum
{
    GenApiCreateFlagDefault = 0x0000, /*!< File is opened with default params as uncompressed XML */
    GenApiCreateFlagXmlZipped = 0x0001  /*!< File is inside a zip-compressed container */
} GenApiNodeMapCreationFlags;


/* NodeMap Properties */
typedef enum
{
    GenApiNodeMapSchemaVersionMajor,
    GenApiNodeMapSchemaVersionMinor,
    GenApiNodeMapXMLVersionMajor,
    GenApiNodeMapXMLVersionMinor,
    GenApiNodeMapXMLVersionSubMinor,
    GenApiNodeMapVendorName,
    GenApiNodeMapModelName
} GenApiNodeMapProperty;


/*!
 * \brief Creates a node map from the given XML at \a pFileName.
 * 
 * \a Flags are
 * - #GenApiCreateFlagDefault for loading uncompressed xml files
 * - #GenApiCreateFlagXmlZipped for loading xml files from a zip container
 * 
 * \param [in] pFileName Path to GenApi XML file to create node map from.
 * \param [in] Flags Configure the creation.
 * \param [in] phNodeMap to struct to be filled with the node map handle.
 * \return
 * - #GenApiSuccess if loaded successfully
 * - #GenApiErrorInvalidArgument if \a pFileName or \a phNodeMap is \b NULL.
 * - #GenApiErrorBadAlloc in out-of-memory conditions.
 * - #GenApiErrorRuntime if load fails.
 */
GENAPI_PCALL(PGenApiCreateNodeMapFromFile)(const char *pFileName, int Flags, GenApiNodeMapHandle *phNodeMap);


/*!
 * \brief Creates a node map from the given XML in \a pBuffer.
 * 
 * \a Flags are
 * - #GenApiCreateFlagDefault for reading uncompressed xml files
 * - #GenApiCreateFlagXmlZipped for reading xml files from a zip container
 * 
 * \param [in] pBuffer Memory buffer containing the XML document.
 * \param [in] BufferSize Size of \a pBuffer in bytes.
 * \param [in] Flags Configure the creation.
 * \param [in] phNodeMap to struct to be filled with the node map handle.
 * \return
 * - #GenApiSuccess if loaded successfully
 * - #GenApiErrorInvalidArgument if \a pBuffer or \a phNodeMap is \b NULL or
 *   BufferSize is 0.
 * - #GenApiErrorBadAlloc in out-of-memory conditions.
 * - #GenApiErrorRuntime if read/parse fails.
 */
GENAPI_PCALL(PGenApiCreateNodeMapFromMemory)(const void *pBuffer, size_t BufferSize, int Flags, GenApiNodeMapHandle *phNodeMap);


/*!
 * \brief Destroys the node map identified by \a hNodeMap and all its nodes.
 * 
 * \param [in] hNodeMap Handle of node map object to delete.
 * \return
 * - #GenApiSuccess if deleted successfully
 * - #GenApiErrorInvalidArgument if \a hNodeMap is \b NULL.
 */
GENAPI_PCALL(PGenApiDestroyNodeMap)(GenApiNodeMapHandle hNodeMap);


/*!
 * \brief Gets a \a Property from the node map identified by \a hNodeMap.
 * 
 * If you want to query the type of the property, simply set \a pType to 
 * #GenApiUnknownType.
 * 
 * For the #GenApiString type a size query will also add the C string 
 * nul-termination. The nul-termination will also be written if enough space is
 * available.
 * 
 * \param [in] hNodeMap Handle of node map object to access.
 * \param [in] Property Identifies the property to access.
 * \param [in,out] pType Non \b NULL hint of which data type to receive; will
 *                  receive written data type.
 * \param [in] pValue Pointer to buffer that is filled with requested data;
 *                  \b NULL for size query.
 * \param [in,out] pValueSize The size in bytes of \a pValue as input; will 
 *                  receive the required (if \a pValue is \b NULL) size or
 *                  actually written size.
 * \return
 * - #GenApiSuccess if property was read successfully.
 * - #GenApiErrorInvalidArgument if \a hNodeMap is \b NULL, \a Property is 
 *      unknown, \a pType is \b NULL or incompatible.
 */
GENAPI_PCALL(PGenApiNodeMapGetProperty)(GenApiNodeMapHandle hNodeMap, GenApiNodeMapProperty Property, GenApiDataType *pType, void *pValue, size_t *pValueSize);


/*!
 * \brief Gets the node with the given \a NodeName from the \a hNodeMap.
 * 
 * \param [in] hNodeMap Handle of node map object to access.
 * \param [in] NodeName Nul-terminated name of the node to retrieve.
 * \param [in] phNode Pointer to be filled with the handle.
 * \return
 * - #GenApiSuccess if property was read successfully.
 * - #GenApiErrorInvalidArgument if \a hNodeMap, \a NodeName or \a phNode is 
 *   \b NULL.
 */
GENAPI_PCALL(PGenApiNodeMapGetNode)(GenApiNodeMapHandle hNodeMap, const char *pNodeName, GenApiNodeHandle *phNode);


/*!
 * \brief Gets a list of all node names in the \a hNodeMap.
 * 
 * The \a pNameList is filled with the names of all nodes the \a hNodeMap. Each 
 * name is 0-terminated. The whole list is also terminated with a 0-byte 
 * resulting in double-0s at the end.
 * 
 * \param [in] hNode Node handle of object to query node names for.
 * \param [in] pNameList Pointer to buffer to be filled with name list; \b NULL
 *              for size query.
 * \param [in,out] pNameListSize The size in bytes of \a pNameList as input; 
 *                  will receive the required (if \a pNameList is \b NULL) size 
 *                  or actually written size.
 * \return
 * - #GenApiSuccess if written successfully
 * - #GenApiErrorInvalidArgument if \a hNodeMap or \a pNameListSize is 
 *   \b NULL.
 * - #GenApiErrorGeneric on other errors.
 * \todo investigate possible exceptions
 */
GENAPI_PCALL(PGenApiNodeMapGetNodeNames)(GenApiNodeMapHandle hNodeMap, char *pNameList, size_t *pNameListSize);


/*!
 * \brief Registers an external Transport Layer Producer port to the 
 *        \a hNodeMap.
 *        
 * Connects a transport layer communication channel to the \a hNodeMap. If
 * \a pReadCallback is \b NULL the port will at most be write only. If 
 * \a pWriteCallback is \b NULL the port will at most be read only. If
 * \a pGetAccessModeCallback is \b NULL the port will be read/write if both
 * \a pReadCallback and \a pWriteCallback are set. It will be read only if only
 * \a PReadCallback is set and write only if \a pWriteCallback is set. If 
 * neither is set the port is not available.
 * 
 * \param [in] hNodeMap Handle of the node map object to connect a port to.
 * \param [in] PortName Name of the port node to connect to (e.g. 'Device' for
 *              a camera/device node map).
 * \param [in] hPort User specified port handle (e.g. personal object's this
 *              pointer).
 * \param [in] pReadCallback Optional pointer to callback for read requests.
 * \param [in] pWriteCallback Optional pointer to callback for write requests.
 * \param [in] pGetAccessModeCallback Optional pointer to callback for access
 *              mode.
 * \return
 * - #GenApiSuccess if connect was successful.
 * 
 */
GENAPI_PCALL(PGenApiNodeMapConnectPort)(GenApiNodeMapHandle hNodeMap, const char *pPortName, GenApiTLPortHandle hPort, PGenApiTLPortRead pReadCallback, PGenApiTLPortWrite pWriteCallback, PGenApiTLPortGetAccessMode pGetAccessModeCallback);


/*!
 * \brief Tells the node map the elapsed time. If the accumulated elapsed time
 * turns greater than or equal to the defined polling time of a node, then the
 * registered callback of the node will be fired.
 *
 * \param [in] hNodeMap Set the handle of the node map object to manipulate.
 * \param [in] ElapsedTime Set a elapsed time in milliseconds.
 * \return
 * - #GenApiSuccess if execution was successful.
 *
 */
GENAPI_PCALL(PGenApiNodeMapPollNodes)(GenApiNodeMapHandle hNodeMap, int64_t ElapsedTime);


/*!
 * \brief Implement your callback function following to this declaration. You can
 *        register your callback function to an arbitrary node calling
 *        \fn PGenApiNodeRegisterCallback and deregister it calling
 *        \fn PGenApiNodeDeregisterCallback.
 *
 * \param [in] hNode Set the handle of a node which calls this callback function.
 * \param [in] pContext Set the pointer to an arbitrary object as a context so that
 *             it can be manipulated in this callback function. You can set \b NULL
 *             if you are sure that you will not use \a pContext in the callback
 *             function.
 * \return Can return but Will not be verified.
 *
 */
GENAPI_PCALL(PGenApiNodeCallback)(GenApiNodeHandle hNode, void* pContext);


/*!
 * \brief Registers a callback function specified by \a pCallback to the node
 *        specified by \a hNode.
 *
 * \param [in]  hNode Set the handle of the node on which the callback is attached.
 * \param [in]  CallbackEvent Set an event that dispatches the callback.
 * \param [in]  pCallback Set the pointer of a callback function to be registered.
 * \param [in]  pContext Set the pointer to an arbitrary object as a context so that
 *              it can be manipulated in the callback function.
 * \param [out] pCallbackToken A callback token will be assigned as a result of
 *              the registration. The token will be used to deregister the callback.
 * \return
 * - #GenApiSuccess if execution was successful.
 *
 */
GENAPI_PCALL(PGenApiNodeRegisterCallback)(GenApiNodeHandle hNode, GenApiNodeCallbackEvent CallbackEvent, PGenApiNodeCallback pCallback, void* pContext, int64_t* pCallbackToken);


/*!
 * \brief Deregisters the callback function that is specified by \a CallbackToken.
 *
 * \param [in] pCallbackToken Set a callback token to deregister the registered
 *             callback.
 * \return
 * - #GenApiSuccess if execution was successful.
 *
 */
GENAPI_PCALL(PGenApiNodeDeregisterCallback)(GenApiNodeHandle hNode, int64_t CallbackToken);


/*!
 * \brief Saves streamable node bvalues, which belongs to a node map
 *        specified by \a hNodeMap, to a file specified by \a pFileName.
 *
 * \param [in] hNodeMap Set a node map handle what you want to interact with.
 * \param [in] pFileName Set a path to the file what you ask the API to store the
 *             node values. The file can be used when you want to load the values
 *             on a node map. The file must be exist in the file system before
 *             calling this function.
 * \return
 * - #GenApiSuccess if execution was successful.
 *
 */
GENAPI_PCALL(PGenApiNodeMapSaveNodeValuesToFile)(GenApiNodeMapHandle hNodeMap, const char* pFileName);


/*!
 * \brief Loads streamable node values, which are stored in a file
 *        specified by \a pFileName, on a node map specified by \a hNodeMap.
 *
 * \param [in] hNodeMap Set a node map handle what you want to interact with.
 * \param [in] pFileName Set a path to the file what you want to load node values from.
 *             The file should be created by calling \fn PGenApiNodeMapSaveNodeValuesToFile.
 * \return
 * - #GenApiSuccess if execution was successful.
 *
 */
GENAPI_PCALL(PGenApiNodeMapLoadNodeValuesFromFile)(GenApiNodeMapHandle hNodeMap, const char* pFileName);


/*!
 * \brief Attaches a buffer specified by \a pBuffer on a node map specified by \a hNodeMap.
 *
 * This function assumes that you are working with a GenTL Producer and especially the
 * following parameters shoud be privided as an outcome of DSGetBufferChunkData.
 *
 * - \a ChunkIDs
 * - \a ChunkOffsets
 * - \a ChunkLengths
 * - \a NumChunks
 *
 * Regarind the following parameters, the n-th element must be mapped to the n-th
 * SingleChunkData_t object, i.e., ChunkIDs[0] must be mapped to SingleChunkData[0].ChunkID.
 *
 * - \a ChunkIDs
 * - \a ChunkOffsets
 * - \a ChunkLengths
 *
 * \param [in] hNodeMap Set a node map handle what you want to interact with.
 * \param [in] pBuffer Set the pointer of a buffer to be parsed.
 * \param [in] pChunkIDs Set the pointer of a list of chunk IDs. A chunk ID is a
 *             numeric representation of the chunk's ChunkID. This information
 *             should be provided by DSGetBufferChunkData.
 * \param [in] pChunkOffsets Set the pointer of a list of chunk offsets. A chunk
 *             offset is the offset of the chunk data from the start of the buffer
 *             (in bytes). This information should be provided by DSGetBufferChunkData.
 * \param [in] pChunkLengths Set the pointer of a list of chunk length. A chunk
 *             length is the size of the given single chunk data (in bytes).
 *             his information should be provided by DSGetBufferChunkData.
 * \param [in] NumChunks Set the number of chunks in the given buffer.
 *             This information should be provided by DSGetBufferChunkData.
 * \return
 * - #GenApiSuccess if execution was successful.
 *
 */
GENAPI_PCALL(PGenApiNodeMapAttachChunkData)(GenApiNodeMapHandle hNodeMap, const void* pBuffer, const uint64_t* pChunkIDs, const int64_t* pChunkOffsets, const uint64_t* pChunkLengths, uint64_t NumChunks);


/*!
 * \brief Updates the chunk data that has been attached to a  node map specified by
 * \a hNodeMap.
 *
 * \param [in] hNodeMap Set a node map handle what you want to interact with.
 * \param [in] pBuffer Set the pointer of a buffer to be parsed.
 * \return
 * - #GenApiSuccess if execution was successful.
 *
 */
GENAPI_PCALL(PGenApiNodeMapUpdateChunkData)(GenApiNodeMapHandle hNodeMap, const void* pBuffer);


/*!
 * \brief Detaches a buffer specified by \a pBuffer from a node map specified by \a hNodeMap.
 *
 * \param [in] hNodeMap Set a node map handle what you want to interact with.
 * \return
 * - #GenApiSuccess if execution was successful.
 *
 */
GENAPI_PCALL(PGenApiNodeMapDetachChunkData)(GenApiNodeMapHandle hNodeMap);


/*!
 * \brief Attaches a buffer specified by \a pBuffer on a node map specified by \a hNodeMap.
 *
 * \param [in] hNodeMap Set a node map handle what you want to interact with.
 * \param [in] pBuffer Set the pointer of an event data to be parsed.
 * \param [in] NumBytes Set the size of event data buffer (in bytes).
 * \param [in] EventID Set the event ID of the event data that you are going to deliver.
 * \return
 * - #GenApiSuccess if execution was successful.
 *
 */
GENAPI_PCALL(PGenApiNodeMapDeliverEventData)(GenApiNodeMapHandle hNodeMap, const void* pBuffer, uint32_t NumBytes, uint64_t EventID);


/*
 * Export
 */

/* The exported functions */
typedef struct
{
    size_t Size;

    PGenApiGetVersion GetVersion;
    PGenApiGetLastErrorString GetLastErrorString;

    PGenApiCreateNodeMapFromFile CreateNodeMapFromFile;
    PGenApiCreateNodeMapFromMemory CreateNodeMapFromMemory;
    PGenApiDestroyNodeMap DestroyNodeMap;
    PGenApiNodeMapConnectPort NodeMapConnectPort;
    PGenApiNodeMapGetNode NodeMapGetNode;
    PGenApiNodeMapGetProperty NodeMapGetProperty;
    PGenApiNodeMapGetNodeNames NodeMapGetNodeNames;
    PGenApiNodeMapPollNodes NodeMapPollNodes;
    PGenApiNodeMapSaveNodeValuesToFile NodeMapSaveNodeValuesToFile;
    PGenApiNodeMapLoadNodeValuesFromFile NodeMapLoadNodeValuesFromFile;
    PGenApiNodeMapAttachChunkData NodeMapAttachChunkData;
    PGenApiNodeMapUpdateChunkData NodeMapUpdateChunkData;
    PGenApiNodeMapDetachChunkData NodeMapDetachChunkData;
    PGenApiNodeMapDeliverEventData NodeMapDeliverEventData;

    PGenApiNodeGetValue NodeGetValue;
    PGenApiNodeSetValue NodeSetValue;
    PGenApiNodeGetProperty NodeGetProperty;
    PGenApiNodeGetLinks NodeGetLinks;
    PGenApiNodeRegisterCallback NodeRegisterCallback;
    PGenApiNodeDeregisterCallback NodeDeregisterCallback;
} GenApiFunctions;


GENAPI_PCALL(POpenGenApi)(GenApiFunctions *pFunctions);


#ifdef GENAPI_C_COMPANY_SUFFIX
#   define GENAPI_C_OPENGENAPI_VER GENAPI_C_TOUNDERSCORE4(OpenGenApiV, GENAPI_C_VERSION_MAJOR, GENAPI_C_VERSION_MINOR, GENAPI_C_COMPANY_SUFFIX)
#else // GENAPI_C_COMPANY_SUFFIX
#   define GENAPI_C_OPENGENAPI_VER GENAPI_C_TOUNDERSCORE3(OpenGenApiV, GENAPI_C_VERSION_MAJOR, GENAPI_C_VERSION_MINOR)
#endif // GENAPI_C_COMPANY_SUFFIX


/*!
 * \brief Opens the GenApi library.
 *
 * This function gives access to all exported GenApi functions. Access to the
 * functions is versioned so you can easily mix different version of the GenApi 
 * in one process (even in the static link case).
 *
 * The GenApiFunctions::Size field is for sanity checking of the used versions.
 *
 * \param [out] pFunctions Pointer to struct to be filled with function pointers.
 */
 /* evaluates e.g. to GENAPI_CALL OpenGenApiV_3_0(GenApiFunctions *pFunctions); */
GENAPI_CALL GENAPI_C_OPENGENAPI_VER(GenApiFunctions *pFunctions);


/*!
 * \brief Shorthand version to load the GenApi you are currently linking against.
 */
#define OpenGenApi GENAPI_C_OPENGENAPI_VER


#endif // GENAPI_C_IFACE_H
