/*
 * SimpleConfigurator.cpp
 *
 * Copyright 2001, Glen Scott. All rights reserved.
 *
 * See the COPYING file for the terms of usage and distribution.
 */
#ifdef _MSC_VER
#   pragma warning( disable : 4702 ) //unreachable code in STL for VC71
#endif

#include "PortabilityImpl.hh"

#ifdef LOG4CPP_HAVE_UNISTD_H
#    include <unistd.h>
#endif
#ifdef LOG4CPP_HAVE_IO_H
#    include <io.h>
#endif

#include <stdio.h>
#include <iostream>
#include <string>
#include <fstream>
#include <stdio.h>

#include <log4cpp/Category.hh>
#include <log4cpp/Appender.hh>
#include <log4cpp/OstreamAppender.hh>
#include <log4cpp/FileAppender.hh>
#include <log4cpp/RollingFileAppender.hh>
#include <log4cpp/Layout.hh>
#include <log4cpp/BasicLayout.hh>
#include <log4cpp/SimpleLayout.hh>
#include <log4cpp/Priority.hh>
#include <log4cpp/NDC.hh>
#include <log4cpp/PatternLayout.hh>
#include <log4cpp/SimpleConfigurator.hh>
#if LOG4CPP_HAVE_SYSLOG
#    include <log4cpp/SyslogAppender.hh>
#endif
#include <log4cpp/RemoteSyslogAppender.hh>
#ifdef _WIN32
#   if !defined(PHARLAP_WIN32) && !defined(UNDER_RTSS)
#    include <log4cpp/NTEventLogAppender.hh>
#   endif
#elif defined(VXWORKS)
#    include <ioLib.h>
#    define _dup dup
#    define _fileno fileno
#else
#    define _fileno ::fileno
#    define _dup    ::dup
#endif

LOG4CPP_NS_BEGIN

void SimpleConfigurator::configure(const std::string& initFileName)
{
    std::ifstream initFile(initFileName.c_str());

    if (!initFile) {
        throw ConfigureFailure(std::string("Config File ") + initFileName + " does not exist or is unreadable");
        //throw INVALID_ARGUMENT_EXCEPTION("Config File '%s' does not exist or is unreadable", initFileName.c_str());
    }

    configure(initFile);
}

void SimpleConfigurator::configure(std::istream& initFile)
{
    std::string nextCommand;
    std::string categoryName;

    while (initFile >> nextCommand) {
        /* skip comment lines */
        if (nextCommand[0] == '#') {
            std::string dummy;
            std::getline(initFile, dummy);
            continue;
        }
        /* stop on missing categoryName */
        if (!(initFile >> categoryName))
            break;

        LOG4CPP_NS::Category& category =
            (categoryName.compare("root") == 0) ?
            LOG4CPP_NS::Category::getRoot() :
            LOG4CPP_NS::Category::getInstance(categoryName);

        if (nextCommand.compare("appender") == 0) {
            std::string layout;
            std::string appenderName;

            if (initFile >> layout >> appenderName) {
                LOG4CPP_NS::Appender* appender;
                if (appenderName.compare("file") == 0) {
                    std::string logFileName;
                    if (!(initFile >> logFileName)) {
                        throw ConfigureFailure("Missing filename for log file logging configuration file for category: " + categoryName);
                        //throw INVALID_ARGUMENT_EXCEPTION("Missing filename for log file logging configuration file for category: %s", categoryName.c_str());
                    }
                    appender = new LOG4CPP_NS::FileAppender(categoryName, logFileName);
                }
                else if (appenderName.compare("rolling") == 0) {
                    std::string logFileName;
                    size_t maxFileSize;
                    unsigned int maxBackupIndex = 1;
                    if (!(initFile >> logFileName)) {
                        throw ConfigureFailure("Missing filename for log file logging configuration file for category: " + categoryName);
                        //throw INVALID_ARGUMENT_EXCEPTION("Missing filename for log file logging configuration file for category: %s", categoryName.c_str());
                    }
                    if (!(initFile >> maxFileSize)) {
                        throw ConfigureFailure("Missing maximum size for log file logging configuration file for category: " + categoryName);
                        //throw INVALID_ARGUMENT_EXCEPTION("Missing maximum size for log file logging configuration file for category: 5S", categoryName.c_str());
                    }
                    if (!(initFile >> maxBackupIndex)) {
                        throw ConfigureFailure("Missing maximum backup index for log file logging configuration file for category: " + categoryName);
                        //throw INVALID_ARGUMENT_EXCEPTION("Missing maximum backup index for log file logging configuration file for category: %s", categoryName.c_str());
                    }
                    appender = new LOG4CPP_NS::RollingFileAppender(categoryName, logFileName, maxFileSize, maxBackupIndex);
                }
                else if (appenderName.compare("console") == 0) {
                    appender =
                        new LOG4CPP_NS::OstreamAppender(categoryName, &std::cout);
                }
                else if (appenderName.compare("stdout") == 0) {
                    appender =
                        new LOG4CPP_NS::FileAppender(categoryName, _dup(_fileno(stdout)));
                }
                else if (appenderName.compare("stderr") == 0) {
                    appender =
                        new LOG4CPP_NS::FileAppender(categoryName, _dup(_fileno(stderr)));
                }
#if LOG4CPP_HAVE_SYSLOG
                else if (appenderName.compare("syslog") == 0) {
                    std::string syslogName;
                    int facility;
                    if (!(initFile >> syslogName)) {
                        throw ConfigureFailure("Missing syslogname for SysLogAppender for category: " + categoryName);
                        //throw INVALID_ARGUMENT_EXCEPTION("Missing syslogname for SysLogAppender for category: %s", categoryName.c_str());
                    }
                    if (!(initFile >> facility)) {
                        facility = LOG_USER;
                    } else {
                        // * 8
                        facility *= 8;
                    }
                    appender =
                        new LOG4CPP_NS::SyslogAppender(categoryName, syslogName, facility);
                }
#endif
#if defined(WIN32) && !defined(PHARLAP_WIN32) && !defined(UNDER_RTSS)
                else if (appenderName.compare("nteventlog") == 0) {
                    std::string source;
                    if (!(initFile >> source)) {
                        throw ConfigureFailure("Missing source for NTEventLogAppender for category: " + categoryName);
                        //throw INVALID_ARGUMENT_EXCEPTION("Missing source for NTEventLogAppender for category: %s", categoryName.c_str());
                    }
                    appender =
                        new LOG4CPP_NS::NTEventLogAppender(categoryName, source);
                }
#endif
                else if (appenderName.compare("remotesyslog") == 0) {
                    std::string syslogName;
                    std::string relayer;
                    int facility;
                    int portNumber;
                    if (!(initFile >> syslogName)) {
                        throw ConfigureFailure("Missing syslogname for SysLogAppender for category: " + categoryName);
                        //throw INVALID_ARGUMENT_EXCEPTION("Missing syslogname for SysLogAppender for category: %s", categoryName.c_str());
                    }
                    if (!(initFile >> relayer)) {
                        throw ConfigureFailure("Missing syslog host for SysLogAppender for category: " + categoryName);
                        //throw INVALID_ARGUMENT_EXCEPTION("Missing syslog host for SysLogAppender for category: %s", categoryName.c_str());
                    }
                    if (!(initFile >> facility)) {
                        facility = LOG_USER;
                    }
                    if (!(initFile >> portNumber)) {
                        portNumber = 514;
                    }
                    appender =
                        new LOG4CPP_NS::RemoteSyslogAppender(categoryName, syslogName, relayer, facility, portNumber);
                }
                else {
                    
                    throw ConfigureFailure("Invalid appender name (" +
                    appenderName +
                    ") in logging configuration file for category: " +
                    categoryName);
                    
                    //throw INVALID_ARGUMENT_EXCEPTION("Invalid appender name (%s) in logging configuration file for category: %s",
                    //    appenderName.c_str(), categoryName.c_str());
                }
                if (layout.compare("basic") == 0)
                    appender->setLayout(new LOG4CPP_NS::BasicLayout());
                else if (layout.compare("simple") == 0)
                    appender->setLayout(new LOG4CPP_NS::SimpleLayout());
                else if (layout.compare("pattern") == 0) {
                    LOG4CPP_NS::PatternLayout *_layout =
                        new LOG4CPP_NS::PatternLayout();
                    initFile >> std::ws; // skip whitespace
                    char pattern[1000];
                    initFile.getline(pattern, 1000);
                    _layout->setConversionPattern(std::string(pattern));
                    appender->setLayout(_layout);
                }
                else {
                    
                    throw ConfigureFailure("Invalid layout (" + layout +
                    ") in logging configuration file for category: " +
                    categoryName);
                    
                    //throw INVALID_ARGUMENT_EXCEPTION("Invalid layout (%s) in logging configuration file for category: %s",
                    //    layout.c_str(), categoryName.c_str());
                }
                category.addAppender(appender);
            }
        }
        else if (nextCommand.compare("priority") == 0) {
            std::string priority;
            if (!(initFile >> priority)) {
                throw ConfigureFailure("Missing priority in logging configuration file for category: " + categoryName);
                //throw INVALID_ARGUMENT_EXCEPTION("Missing priority in logging configuration file for category: %s", categoryName.c_str());
            }

            try {
                category.setPriority(LOG4CPP_NS::Priority::getPriorityValue(priority));
            }
            catch (std::invalid_argument) {
                throw ConfigureFailure("Invalid priority ("+priority+") in logging configuration file for category: "+categoryName);
                //throw INVALID_ARGUMENT_EXCEPTION("Invalid priority (%s) in logging configuration file for category: %s",
                  //  priority.c_str(), categoryName.c_str());
            }
        }
        else if (nextCommand.compare("category") == 0) {
            /*
              This command means we should "refer" to the category
              (in order to have it created). We've already done this
              in common setup code for all commands.
              */
        }
        else {
            throw ConfigureFailure("Invalid format in logging configuration file. Command: " + nextCommand);
            //throw INVALID_ARGUMENT_EXCEPTION("Invalid format in logging configuration file. Command: %s", nextCommand.c_str());

        }
    }
}
LOG4CPP_NS_END



