#include <iostream>
#include <string.h>
#include <log4cpp/Log4cppAdapter.hh>
#include <log4cpp/CategoryAdapter.hh>
#include <log4cpp/PropertyConfigurator.hh>
#include <log4cpp/NDC.hh>
#include <log4cpp/PatternLayout.hh>
#include <log4cpp/Win32DebugAppender.hh>

#include <GenICamVersion.h>
#include <log4cpp/Extensions/Utilities.h>
#include <fstream>
#include <cassert>

LOG4CPP_NS_BEGIN

Log4cppAdapter::Log4cppAdapter() :
  mLoggerFactoryName("LOG4CPP"),
  mHasBeenConfigured(false)
{
}

Log4cppAdapter::~Log4cppAdapter()
{
  
}


const char* Log4cppAdapter::GetLoggerFactoryName()
{
  return mLoggerFactoryName.c_str();
}

static bool PreProcessConfiguration(std::ostream& os, std::istream& is)
{
    while (is.good()) {
        std::string LineBuffer;
        std::getline(is, LineBuffer);
        if (!LineBuffer.empty())
        {
            if (LineBuffer[0] != '#') {
                // Replace Environment variables within the content of the line
                // ignoring lines which are comments (they have a # in the first column
                ReplaceEnvironmentVariables(LineBuffer);
            }

            os << LineBuffer << "\n";
        }
        if (!os)
        {
            return false;
        }
  }

  return true;
}
void RemoveAllAppenders()
{
    std::vector<LOG4CPP_NS::Category *> *pAllCategies = LOG4CPP_NS::Category::getCurrentCategories();
    for (std::vector<LOG4CPP_NS::Category *>::iterator it = pAllCategies->begin(); it != pAllCategies->end(); it++)
    {
        (*it)->removeAllAppenders();
    }
    delete pAllCategies;

}
void Log4cppAdapter::ConfigureFromString(const char* configString)
{
  try
  {
    
    if (mHasBeenConfigured)
    {
        //Category::getRoot().removeAllAppenders();
        RemoveAllAppenders();
    }
    
    std::stringstream buffer;
    buffer << configString;

    std::stringstream preProcessedBuffer;
    if (!PreProcessConfiguration(preProcessedBuffer, buffer)) {
      mHasBeenConfigured = false;
    }
    else
    {
      std::istream& input(preProcessedBuffer);
      PropertyConfigurator::configure(input);
      mHasBeenConfigured = true;
    }
  
  }
  catch (const std::exception &ex)
  {
    std::cout << ex.what() << std::endl;
  }
  catch (...) // no exception should cross the dll boundary
  {
    std::cout << "Unknown exception caught during from-string configuration" << std::endl;
    assert(! "Unknown excetion caught during from-string config" );
  }
}

void Log4cppAdapter::ConfigureDefault()
{
  if (mHasBeenConfigured)
  {
      //Category::getRoot().removeAllAppenders();
      RemoveAllAppenders();
  }

  std::stringstream logConfig;
  logConfig << "$(" << GENICAM_LOG_CONFIG_VERSION << ")";
  std::string path1 = logConfig.str();

  std::stringstream genICamRoot;
#ifdef _WIN32
  genICamRoot << "$(" << GENICAM_ROOT << ")/log/config/DefaultLogging.properties";
#else
  genICamRoot << "$(" << GENICAM_ROOT << ")/log/config-unix/DefaultLogging.properties";	
#endif
  std::string path2 = genICamRoot.str();

  std::string configuration;
  try
  {
    if (ReplaceEnvironmentVariables(path1))
    {
      // Default 1: from ENV: GENICAM_LOG_CONFIG_VERSION
      std::ifstream inFile(path1.c_str());
      if (inFile.is_open())
      {
        configuration =std::string((std::istreambuf_iterator<char>(inFile)), std::istreambuf_iterator<char>());
      }
    }
    if (configuration.empty() && ReplaceEnvironmentVariables(path2))
    {
      // Default 2: from ENV: GENICAM_ROOT
      std::ifstream inFile(path2.c_str());
      if (inFile.is_open())
      {
        configuration =std::string((std::istreambuf_iterator<char>(inFile)), std::istreambuf_iterator<char>());
      }
    }
    if (configuration.empty())
    {
      // Default 3: from hard coded configuration
      std::stringstream ss;
#ifdef _WIN32
      ss << "log4cpp.rootCategory=ERROR, Win32Debug" << std::endl;
      ss << "log4cpp.category.GenApi=ERROR, Win32Debug" << std::endl;
      ss << "log4cpp.appender.Win32Debug=org.apache.log4cpp.Win32DebugAppender" << std::endl;
      ss << "log4cpp.appender.Win32Debug.layout=org.apache.log4cpp.PatternLayout" << std::endl;
      ss << "log4cpp.appender.Win32Debug.layout.ConversionPattern==>LOG %x: %c : %m%n" << std::endl;
#else
      ss << "log4cpp.rootCategory=ERROR, Console" << std::endl;
      ss << "log4cpp.category.GenApi=ERROR, Console" << std::endl;
      ss << "log4cpp.appender.Console=org.apache.log4cpp.ConsoleAppender" << std::endl;
      ss << "log4cpp.appender.Console.layout=org.apache.log4cpp.PatternLayout" << std::endl;
      ss << "log4cpp.appender.Console.layout.ConversionPattern==>LOG %x: %c : %m%n" << std::endl;
#endif
      configuration =ss.str();
    }
    ConfigureFromString(configuration.c_str());

  }
  catch (const std::exception &ex)
  {
    std::cout << ex.what() << std::endl;
  }
  catch (...) // no exception should cross the dll boundary
  {
    std::cout << "Unknown exception caught during default configuration" << std::endl;
    assert(! "Unknown excetion caught during config" );
  }
}

// Creates a new logger
// The caller has to remember if this logger has been called yet
GENICAM_NAMESPACE::ILogger* Log4cppAdapter::GetLogger(const char* name)
{
  try
  {
      if (!mHasBeenConfigured)
      {
          ConfigureDefault();
      }

    GENICAM_NAMESPACE::ILogger* pLogger = new CategoryAdapter(LOG4CPP_NS::Category::getInstance(std::string(name)));
    return pLogger;
  }
  catch (const std::exception &ex)
  {
    std::cout << ex.what() << std::endl;
  }
  catch (...) // no exception should cross the dll boundary
  {
    std::cout << "Unknown exception caught" << std::endl;
  }
  return NULL;
}

bool Log4cppAdapter::Exist(const char* name)
{
  try
  {
      if (!mHasBeenConfigured)
      {
          ConfigureDefault();
      }
    LOG4CPP_NS::Category* pCat = LOG4CPP_NS::Category::exists(std::string(name));
    return (pCat != NULL);
  }
  catch (const std::exception &ex)
  {
    std::cout << ex.what() << std::endl;
  }
  catch (...) // no exception should cross the dll boundary
  {
    std::cout << "Unknown exception caught" << std::endl;
  }
  return false;
}

void Log4cppAdapter::PushIndent()
{
  try
  {
      if (!mHasBeenConfigured)
      {
          ConfigureDefault();
      }

    LOG4CPP_NS::NDC::push("  ");
  }
  catch (const std::exception &ex)
  {
    std::cout << ex.what() << std::endl;
  }
  catch (...) // no exception should cross the dll boundary
  {
    std::cout << "Unknown exception caught" << std::endl;
  }
}

void Log4cppAdapter::PopIndent()
{
  try
  {
      if (!mHasBeenConfigured)
      {
          ConfigureDefault();
      }

    LOG4CPP_NS::NDC::pop();
  }
  catch (const std::exception &ex)
  {
    std::cout << ex.what() << std::endl;
  }
  catch (...) // no exception should cross the dll boundary
  {
    std::cout << "Unknown exception caught" << std::endl;
  }
}



LOG4CPP_NS_END

